#!/bin/sh

# This script contains helper functions used in dep_check.sh and dep_support_collection.sh

# Copyright © 2022-2024 Audinate Pty Ltd ACN 120 828 006 (Audinate). All rights reserved.
#
#
# 1.	Subject to the terms and conditions of this Licence, Audinate hereby grants you a worldwide, non-exclusive,
#		no-charge, royalty free licence to copy, modify, merge, publish, redistribute, sublicense, and/or sell the
#		Software, provided always that the following conditions are met:
#		1.1.	the Software must accompany, or be incorporated in a licensed Audinate product, solution or offering
#				or be used in a product, solution or offering which requires the use of another licensed Audinate
#				product, solution or offering. The Software is not for use as a standalone product without any
#				reference to Audinate's products;
#		1.2.	the Software is provided as part of example code and as guidance material only without any warranty
#				or expectation of performance, compatibility, support, updates or security; and
#		1.3.	the above copyright notice and this License must be included in all copies or substantial portions
#				of the Software, and all derivative works of the Software, unless the copies or derivative works are
#				solely in the form of machine-executable object code generated by the source language processor.
#
# 2.	TO THE EXTENT PERMITTED BY APPLICABLE LAW, THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
#		EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
#		PURPOSE AND NONINFRINGEMENT.
#
# 3.	TO THE FULLEST EXTENT PERMITTED BY APPLICABLE LAW, IN NO EVENT SHALL AUDINATE BE LIABLE ON ANY LEGAL THEORY
#		(INCLUDING, WITHOUT LIMITATION, IN AN ACTION FOR BREACH OF CONTRACT, NEGLIGENCE OR OTHERWISE) FOR ANY CLAIM,
#		LOSS, DAMAGES OR OTHER LIABILITY HOWSOEVER INCURRED.  WITHOUT LIMITING THE SCOPE OF THE PREVIOUS SENTENCE THE
#		EXCLUSION OF LIABILITY SHALL INCLUDE: LOSS OF PRODUCTION OR OPERATION TIME, LOSS, DAMAGE OR CORRUPTION OF
#		DATA OR RECORDS; OR LOSS OF ANTICIPATED SAVINGS, OPPORTUNITY, REVENUE, PROFIT OR GOODWILL, OR OTHER ECONOMIC
#		LOSS; OR ANY SPECIAL, INCIDENTAL, INDIRECT, CONSEQUENTIAL, PUNITIVE OR EXEMPLARY DAMAGES, ARISING OUT OF OR
#		IN CONNECTION WITH THIS AGREEMENT, ACCESS OF THE SOFTWARE OR ANY OTHER DEALINGS WITH THE SOFTWARE, EVEN IF
#		AUDINATE HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH CLAIM, LOSS, DAMAGES OR OTHER LIABILITY.
#
# 4.	APPLICABLE LEGISLATION SUCH AS THE AUSTRALIAN CONSUMER LAW MAY APPLY REPRESENTATIONS, WARRANTIES, OR CONDITIONS,
#		OR IMPOSES OBLIGATIONS OR LIABILITY ON AUDINATE THAT CANNOT BE EXCLUDED, RESTRICTED OR MODIFIED TO THE FULL
#		EXTENT SET OUT IN THE EXPRESS TERMS OF THIS CLAUSE ABOVE "CONSUMER GUARANTEES".	 TO THE EXTENT THAT SUCH CONSUMER
#		GUARANTEES CONTINUE TO APPLY, THEN TO THE FULL EXTENT PERMITTED BY THE APPLICABLE LEGISLATION, THE LIABILITY OF
#		AUDINATE UNDER THE RELEVANT CONSUMER GUARANTEE IS LIMITED (WHERE PERMITTED AT AUDINATE'S OPTION) TO ONE OF
#		FOLLOWING REMEDIES OR SUBSTANTIALLY EQUIVALENT REMEDIES:
#		4.1.	THE REPLACEMENT OF THE SOFTWARE, THE SUPPLY OF EQUIVALENT SOFTWARE, OR SUPPLYING RELEVANT SERVICES AGAIN;
#		4.2.	THE REPAIR OF THE SOFTWARE;
#		4.3.	THE PAYMENT OF THE COST OF REPLACING THE SOFTWARE, OF ACQUIRING EQUIVALENT SOFTWARE, HAVING THE RELEVANT
#				SERVICES SUPPLIED AGAIN, OR HAVING THE SOFTWARE REPAIRED.
#
# 5.	This License does not grant any permissions or rights to use the trade marks (whether registered or unregistered),
#		the trade names, or product names of Audinate.
#
# 6.	If you choose to redistribute or sell the Software you may elect to offer support, maintenance, warranties,
#		indemnities or other liability obligations or rights consistent with this License. However, you may only act on
#		your own behalf and must not bind Audinate. You agree to indemnify and hold harmless Audinate, and its affiliates
#		from any liability claimed or incurred by reason of your offering or accepting any additional warranty or additional
#		liability.
#

# colors
RED_COLOR="\e[01;31m"
GREEN_COLOR="\e[01;32m"
YELLOW_COLOR="\e[01;33m"
BLUE_COLOR="\e[01;34m"
END_COLOR="\e[0m"

TAB()     { printf "\t"; }
red()     { printf "$RED_COLOR $* $END_COLOR"; }
green()   { printf "$GREEN_COLOR $* $END_COLOR"; }
blue()    { printf "$BLUE_COLOR $* $END_COLOR"; }
yellow()  { printf "$YELLOW_COLOR $* $END_COLOR"; }

logerr()  { echo "[  $(red ERROR)] $1"; }
logwarn() {	echo "[$(yellow WARNING)] $1"; }
loginfo() {	echo "[   $(blue INFO)] $1"; }
logok()   { echo "[     $(green OK)] $1"; }


# if we ever need to check if sh is busybox
#exe=`exec 2>/dev/null; readlink "/proc/$$/exe"`
#if [ "$exe" = "/bin/busybox" ]; then
#	...

command_available()
{
    command -v > /dev/null 2> /dev/null "$1"
    RETVAL=$?
    if [ "$RETVAL" -eq 0 ]; then # available
        RETVAL="available"
    else
        RETVAL="absent"
    fi
}

get_config()
{
    FILE=""
    EXTENSION="txt"
    GUNZIPPED_FILE=0
    if [ -f "/proc/config.gz" ]; then
        FILE="gunzipped_config"
        gunzip -c "/proc/config.gz" > $FILE
        RETVAL=$?
        if [ "$RETVAL" -eq 0 ]; then
            GUNZIPPED_FILE=1
        else
            # gunzip is unavailable or returned an error
            # In this case just use the .gz file
            rm "$FILE"
            FILE="/proc/config.gz"
            EXTENSION="gz"
        fi
    else
        if [ -f "/boot/config-$(uname -r)" ]; then
            FILE="/boot/config-$(uname -r)"
        elif [ -f "/boot/config" ]; then
            FILE="/boot/config"
        elif [ -f "/lib/modules/$(uname -r)/build/.config" ]; then
            FILE="/lib/modules/$(uname -r)/build/.config"
        fi
    fi
}

# kernel_info(): find kernel information in many possible places. 
# $1 is the directory path to store the copy of kernel config.
#
# If $1 is empty, display kernel config on stdout.
kernel_info()
{
    get_config # returns FILE, EXTENSION and GUNZIPPED_FILE
    if [ -z "$FILE" ] && [ -z "$1" ]; then
        # Return an empty config
        echo ""
        return
    fi

    KERNEL_CONFIG=$(cat "$FILE")

    # If FILE is still gzipped:
    #
    # - KERNEL_CONFIG will be nonempty and consist of binary data (if
    #   FILE has a nonzero size)
    # - If we're copying FILE, checking KERNEL_CONFIG for nonemptiness
    #   is still OK
    # - If however we're displaying KERNEL_CONFIG instead, output an
    #   empty string
    if [ -n "$KERNEL_CONFIG" ]; then
        if [ -n "$1" ]; then
            if [ -d "$1" ]; then
                # only does copy operation to the specified destination directory
                KERNEL_INFO_DEST="$1/kernel_config.$EXTENSION"

                if ! cp "$FILE" "$KERNEL_INFO_DEST"; then
                    logwarn "copy of $FILE to $1 failed"
                fi
            else
                logwarn "copy of $FILE to $1: $1 is not a valid path"
            fi
        else
            # Just output $KERNEL_CONFIG
            if [ "$EXTENSION" = "gz" ]; then
                echo ""
            else
                echo "$KERNEL_CONFIG"
            fi
        fi
    else
        if [ -z "$1" ]; then
            echo "$KERNEL_CONFIG"
        fi
    fi

    # If FILE is gunzipped, it's only temporary so delete it here
    if [ "$GUNZIPPED_FILE" -eq 1 ]; then
        rm "$FILE"
    fi
}

# Simple json value extractor. Only works for the cases where the key and
# value are on the same line.
# $1 is the key to extract
# $2 is the json file
dante_json_value()
{
    key=$1
    json_file=$2

    # extract value
    v=$(grep -n \"$key\" "$json_file" | cut -d: -f3 | sed -re 's/("|,)//g')

    # echo removes leading and trailing spaces
    echo $v
}

# Array routines
array_length() { echo "$1" | wc -w; }

array_report_duplicates() {
    DUPLICATES=$(echo "$1" | tr ' ' '\n' | sort | uniq -d | sort -u | tr '\n' ' ')
    DUPLICATES=${DUPLICATES%?} # Remove trailing space
    echo "$DUPLICATES"
}

array_remove_duplicates() {
    ARRAY_NO_DUPLICATES=$(echo "$1" | tr ' ' '\n' | sort -u | tr '\n' ' ')
    ARRAY_NO_DUPLICATES=${ARRAY_NO_DUPLICATES%?}
    echo "$ARRAY_NO_DUPLICATES"
}

# Given two arrays, report entries found in both
# Assumes that each of them don't have duplicates to begin with
array_report_overlap() {
    COMBINED="$1 $2"
    array_report_duplicates "$COMBINED"
}

# Given two arrays, remove from the first array any entries which are present in the second
array_remove_entries() {
    if [ "$2" = "" ]; then
        echo "$1"
    else
        ENTRIES_TO_REMOVE=$(echo "$2" | sed 's/ /\\|/g')
        NEW=$(echo "$1" | tr ' ' '\n' | grep -vw "$ENTRIES_TO_REMOVE" | tr '\n' ' ')
        NEW=${NEW%?}
        echo "$NEW"
    fi
}

# Remove a specified number of entries from the end of an array
array_trim_end() {
    LENGTH=$(array_length "$1")
    ENTRIES_TO_TRIM="$2"

    NEW=$(echo "$1" | tr ' ' '\n' | head -n $((LENGTH-ENTRIES_TO_TRIM)) | tr '\n' ' ')
    NEW=${NEW%?}
    echo "$NEW"
}

# Given an array JSON field, return its elements as a string separated by
# spaces.
#
# This function is able to handle the following JSON node styles - e.g. for
# the "interfaces" field:
#
# {
#   "interfaces":
#     [
#       "eth0",
#       "eth1"
#     ],
#   "interfaces": [
#     "eth1",
#     "eth2"
#   ],
#   "interfaces": [ "eth2", "eth3"
#   ],
#   "interfaces": [ "eth3", "eth4" ]
# }
#
get_array_entries() {
    field_name="$1"
    result=""

    start_line=$(grep -n "$field_name" "$DANTE_JSON" | cut -d: -f1)
    # if the field is missing, return empty
    if [ -z "$start_line" ]; then
        echo "$result"
        return
    fi

    # If the passed in field_name is "interfaces", this will be one of (e.g.):
    # - "interfaces":
    # - "interfaces": [
    # - "interfaces": [ "eth0"
    # - "interfaces": [ "eth0" ],
    start_line_text=$(grep "$field_name" "$DANTE_JSON" | head -n1)

    # remove trailing whitespaces (spaces or tabs)
    trimmed_line="${start_line_text%"${start_line_text##*[![:space:]]}"}"

    ends_with_bracket_or_bracket_comma=false
    [ "${trimmed_line%"],"}" != "$trimmed_line" ] || [ "${trimmed_line%]}" != "$trimmed_line" ] && ends_with_bracket_or_bracket_comma=true

    ends_with_double_quote_or_quote_comma=false
    [ "${trimmed_line%"\","}" != "$trimmed_line" ] || [ "${trimmed_line%\"}" != "$trimmed_line" ] && ends_with_double_quote_or_quote_comma=true

    # if the string ends with ']' or '],' or '"' or '",', start parsing from start_line
    if [ "$ends_with_bracket_or_bracket_comma" = true ] || [ "$ends_with_double_quote_or_quote_comma" = true ]; then

        # remove everything before the opening square bracket
        temp_var="${trimmed_line#*[}"

        # remove the closing square bracket
        temp_var="${temp_var%]*}"

        # iterate through each comma-separated value
        while [ -n "$temp_var" ]; do
            # remove everything before the first double quote, including the double quote itself
            current="${temp_var#*\"}"

            # remove everything after the first double quote, including the double quote itself
            current="${current%%\"*}"

            # add the current interface and newline to the result variable
            result="${result}${current}"" "

            # remove the current interface from temp_var for the next iteration
            temp_var="${temp_var#*\"$current\"}"

            # if there is a comma, remove it
            if [ "${temp_var%%,*}" != "$temp_var" ]; then
                temp_var="${temp_var#*,}"
            else
                break
            fi
        done

        # Put all elements into a single space-separated string
        result=$(echo $result | tr ' ' '\n' | sed 's/,//g' | tr '\n' ' ')

    # in all other cases, we need to continue parsing between start_line+n and the next
    # ']' or '],' or ']<number of spaces>'
    else
        # if the line ends with ':' or ':<number of spaces>", we need to increment
        # start_line by two as the opening [ bracket will be on the next line,
        # which we want to skip over
        if [ "${trimmed_line%:}" != "$trimmed_line" ]; then
            start_line=$((start_line+2))

        # otherwise just move the pointer to the next line
        else
            start_line=$((start_line+1))
        fi

        # find the next closing ] bracket
        end_line=$(tail -n +${start_line} ${DANTE_JSON} | grep -n -e '],'  -e ']' | head -n1 | cut -d: -f1)
        end_line=$((end_line-1))

        # Put all elements into a single space-separated string
        result=$(tail -n +${start_line} ${DANTE_JSON} | head -n${end_line} | sed -re 's/(\s|"|,|])//g')
    fi

    echo "$result"
}

# starts_with
#
# This function checks if the first string starts with the second string.
#
# Usage: starts_with "string" "prefix"
# 
# Arguments:
#   $1: The first string to be checked.
#   $2: The prefix to check against the first string.
#
# Returns:
#   0 (true) if the first string starts with the prefix.
#   1 (false) otherwise.
#
starts_with() {
    case "$1" in
        "$2"*) 
            return 0
        ;;
        *)  
            return 1
        ;;
    esac
}

#
# Copyright © 2022-2024 Audinate Pty Ltd ACN 120 828 006 (Audinate). All rights reserved.
#

